'use strict';

/* --------------------------------------------------------------
 state.js 2016-06-20
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Handles the table state for filtering, pagination and sorting.
 *
 * This controller will update the window history with the current state of the table. It reacts
 * to specific events such as filtering, pagination and sorting changes. After the window history
 * is updated the user will be able to navigate forth or backwards.
 *
 * Notice #1: This module must handle the window's pop-state events and not other modules because
 * this will lead to unnecessary code duplication and multiple AJAX requests.
 *
 * Notice #1: The window state must be always in sync with the URL for easier manipulation.
 */
gx.controllers.module('state', [jse.source + '/vendor/jquery-deparam/jquery-deparam.min.js'], function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES
    // ------------------------------------------------------------------------

    /**
     * Module Selector
     *
     * @type {jQuery}
     */

    var $this = $(this);

    /**
     * Module Instance
     *
     * @type {Object}
     */
    var module = {};

    /**
     * Window History Support
     *
     * @type {Boolean}
     */
    var historySupport = jse.core.config.get('history');

    // ------------------------------------------------------------------------
    // FUNCTIONS
    // ------------------------------------------------------------------------

    /**
     * Get parsed state from the URL GET parameters.
     *
     * @return {Object} Returns the table state.
     */
    function _getState() {
        return $.deparam(window.location.search.slice(1));
    }

    /**
     * Set the state to the browser's history.
     *
     * The state is stored for enabling back and forth navigation from the browser.
     *
     * @param {Object} state Contains the new table state.
     */
    function _setState(state) {
        var url = window.location.origin + window.location.pathname + '?' + $.param(state);
        window.history.pushState(state, '', url);
    }

    /**
     * Update page navigation state.
     *
     * @param {jQuery.Event} event jQuery event object.
     * @param {Object} pagination Contains the DataTable pagination info.
     */
    function _onPageChange(event, pagination) {
        var state = _getState();

        state.page = pagination.page + 1;

        _setState(state);
    }

    /**
     * Update page length state.
     *
     * @param {jQuery.Event} event jQuery event object.
     * @param {Number} length New page length.
     */
    function _onLengthChange(event, length) {
        var state = _getState();

        state.page = 1;
        state.length = length;

        _setState(state);
    }

    /**
     * Update filter state.
     *
     * @param {jQuery.Event} event jQuery event object.
     * @param {Object} filter Contains the filtering values.
     */
    function _onFilterChange(event, filter) {
        var state = _getState();

        state.page = 1;
        state.filter = filter;

        _setState(state);
    }

    /**
     * Update sort state.
     *
     * @param {jQuery.Event} event jQuery event object.
     * @param {Object} sort Contains column sorting info {index, name, direction}.
     */
    function _onSortChange(event, sort) {
        var state = _getState();

        state.sort = (sort.direction === 'desc' ? '-' : '+') + sort.name;

        _setState(state);
    }

    /**
     * Set the correct table state.
     *
     * This method will parse the new popped state and apply it on the table. It must be the only place where this
     * happens in order to avoid multiple AJAX requests and data collisions.
     *
     * @param {jQuery.Event} event
     */
    function _onWindowPopState(event) {
        var state = event.originalEvent.state || {};

        if (state.page) {
            $this.find('.page-navigation select').val(state.page);
            $this.DataTable().page(parseInt(state.page) - 1);
        }

        if (state.length) {
            $this.find('.page-length select').val(state.length);
            $this.DataTable().page.len(parseInt(state.length));
        }

        if (state.sort) {
            var _$this$DataTable$init = $this.DataTable().init(),
                columns = _$this$DataTable$init.columns;

            var direction = state.sort.charAt(0) === '-' ? 'desc' : 'asc';
            var name = state.sort.slice(1);
            var index = 1; // Default Value

            var _iteratorNormalCompletion = true;
            var _didIteratorError = false;
            var _iteratorError = undefined;

            try {
                for (var _iterator = columns[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
                    var column = _step.value;

                    if (column.name === name) {
                        index = columns.indexOf(column);
                        break;
                    }
                }
            } catch (err) {
                _didIteratorError = true;
                _iteratorError = err;
            } finally {
                try {
                    if (!_iteratorNormalCompletion && _iterator.return) {
                        _iterator.return();
                    }
                } finally {
                    if (_didIteratorError) {
                        throw _iteratorError;
                    }
                }
            }

            $this.DataTable().order([index, direction]);
        }

        if (state.filter) {
            // Update the filtering input elements.
            for (var _column in state.filter) {
                var value = state.filter[_column];

                if (value.constructor === Array) {
                    value = value.join('||'); // Join arrays into a single string.
                }

                $this.DataTable().column(_column + ':name').search(value);
            }
        }

        $this.DataTable().draw(false);
    }

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {
        if (historySupport) {
            $this.on('datatable_custom_pagination:page_change', _onPageChange).on('datatable_custom_pagination:length_change', _onLengthChange).on('datatable_custom_sorting:change', _onSortChange).on('orders_overview_filter:change', _onFilterChange);

            $(window).on('popstate', _onWindowPopState);
        }

        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
